// main.cpp
// A minimalist C++/WTL web browser using the WebView2 engine.

// Target Windows 10 and later.
#define _WIN32_WINNT 0x0A00
#include <windows.h>
#include <atlbase.h>
#include <atlcom.h>
#include <atlapp.h>
#include <atlwin.h>
#include <atlcrack.h>
#include <atlframe.h>
#include <atlctrls.h>
#include <wrl.h> // For Microsoft::WRL::ComPtr
#include <wil/com.h> // For wil::com_ptr

// Include the WebView2 header.
#include "WebView2.h"

// Global Module object for WTL
CAppModule _Module;

// Main Window Class
class CMainWindow : public CFrameWindowImpl<CMainWindow>
{
public:
    // WTL window class declaration
    DECLARE_WND_CLASS(L"MinimalBrowser")

    // WebView2 specific members
    wil::com_ptr<ICoreWebView2Controller> m_controller;
    wil::com_ptr<ICoreWebView2> m_webview;

    // UI Elements
    CEdit m_addressBar;

    // WTL Message Map
    BEGIN_MSG_MAP(CMainWindow)
        MSG_WM_CREATE(OnCreate)
        MSG_WM_SIZE(OnSize)
        COMMAND_ID_HANDLER(ID_GO, OnGo) // Handle "Go" button clicks
        CHAIN_MSG_MAP(CFrameWindowImpl<CMainWindow>)
    END_MSG_MAP()

    // Window Creation Handler
    LRESULT OnCreate(LPCREATESTRUCT lpCreateStruct)
    {
        // 1. Create the Address Bar (an Edit control)
        m_hWndClient = m_addressBar.Create(m_hWnd, rcDefault, L"https://www.google.com", 
            WS_CHILD | WS_VISIBLE | WS_CLIPSIBLINGS | ES_AUTOHSCROLL, 0, 12345); // Using a dummy ID

        // This sets the CMainWindow to handle messages for the frame.
        SetMsgHandled(FALSE);

        // 2. Initialize the WebView2 Environment
        // This is an asynchronous operation.
        CreateCoreWebView2EnvironmentWithOptions(nullptr, nullptr, nullptr,
            Callback<ICoreWebView2CreateCoreWebView2EnvironmentCompletedHandler>(
                [this](HRESULT result, ICoreWebView2Environment* env) -> HRESULT {
                    
                    // 3. Create the WebView2 Controller
                    // This embeds the browser control in our window.
                    env->CreateCoreWebView2Controller(m_hWnd, Callback<ICoreWebView2CreateCoreWebView2ControllerCompletedHandler>(
                        [this](HRESULT result, ICoreWebView2Controller* controller) -> HRESULT {
                            if (controller != nullptr) {
                                m_controller = controller;
                                m_controller->get_CoreWebView2(&m_webview);
                            }

                            // 4. Set WebView2 properties
                            // Attach the WebView2 control to the window size.
                            RECT bounds;
                            GetClientRect(&bounds);
                            m_controller->put_Bounds(bounds);
                            
                            // Adjust for the address bar.
                            OnSize(0, bounds.right, bounds.bottom);

                            // 5. Navigate to the initial page.
                            m_webview->Navigate(L"https://www.google.com");

                            return S_OK;
                        }).Get());
                    return S_OK;
                }).Get());

        return 0;
    }

    // Window Resize Handler
    void OnSize(UINT nType, int cx, int cy)
    {
        if (m_controller == nullptr) return;

        // Position the address bar at the top.
        const int addressBarHeight = 24;
        ::SetWindowPos(m_addressBar, NULL, 0, 0, cx, addressBarHeight, SWP_NOZORDER);

        // Resize the WebView2 control to fill the rest of the window.
        RECT wvBounds = { 0, addressBarHeight, cx, cy };
        m_controller->put_Bounds(wvBounds);
    }
    
    // Handler for navigation (e.g., from an address bar)
    LRESULT OnGo(WORD wNotifyCode, WORD wID, HWND hWndCtl, BOOL& bHandled)
    {
        if (m_webview == nullptr) return 0;
        
        // Get text from the address bar
        wchar_t url[2048];
        m_addressBar.GetWindowText(url, 2048);

        // Navigate the webview to the URL
        m_webview->Navigate(url);
        
        return 0;
    }
};

// Main application entry point
int WINAPI wWinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, PWSTR pCmdLine, int nCmdShow)
{
    // Initialize COM for WebView2
    CoInitializeEx(NULL, COINIT_APARTMENTTHREADED);

    // Initialize WTL
    _Module.Init(NULL, hInstance);

    // Message loop
    CMessageLoop msgLoop;
    _Module.AddMessageLoop(&msgLoop);

    // Create the main window
    CMainWindow win;
    if (win.Create(NULL, CWindow::rcDefault, L"Minimal Browser", WS_OVERLAPPEDWINDOW | WS_VISIBLE) == NULL)
    {
        ATLTRACE(L"Main window creation failed!\n");
        return 0;
    }

    // Run the message loop
    int nRet = msgLoop.Run();

    // Uninitialize WTL and COM
    _Module.RemoveMessageLoop();
    _Module.Term();
    CoUninitialize();

    return nRet;
}